function getServerURL() {
    var serverURL = '';
    if (typeof app !== 'undefined' && app && typeof app == 'object' && (app?.getSetting || false) && typeof app.getSetting === 'function') {
        serverURL = app.getSetting('server_url') || '';
        if (serverURL.endsWith('/')) serverURL = serverURL.slice(0, -1);
    }
    if (!serverURL) {
        serverURL = window.location.origin;
    }
    return serverURL;
}
function renderActionResult(chat_id, action, author_pk) {
    console.log("renderActionResult", chat_id, action, author_pk);
    /* 
        Action button listeners are abstract so that new actions can be added 
        without changing this script, with the exception of THIS FUNCTION. 
    */
    // Validate inputs
    if (!chat_id || isNaN(chat_id * 1) || (chat_id * 1) === 0) { // can be a chat or a thread id
        throw new Error('Invalid chat_id');
    }
    if (!action || typeof action !== 'string' || action.length < 1) {
        throw new Error('Invalid action');
    }
    if (!author_pk || typeof author_pk !== 'string' || author_pk.length < 1) {
        throw new Error('Invalid author_pk');
    }
    // Perform action (edit this section when adding new actions)
    switch (action.toLowerCase()) {
        case 'bookmark':
        case 'unbookmark':
            return; // do nothing. Bookmarking is handled by the server and the bookmark button.
        case 'block': // add class "blocked" to all chats and theads from the author based on the author_pk.
            $(`.thread[data-author-public-key="${author_pk}"]`).addClass('blocked');
            $(`.chat[data-author-pk="${author_pk}"]`).addClass('blocked');
            break;
        case 'unblock': // remove class "blocked" from all chats and theads from the author based on the author_pk.
            $(`.thread[data-author-public-key="${author_pk}"]`).removeClass('blocked');
            $(`.chat[data-author-pk="${author_pk}"]`).removeClass('blocked');
            break;
        case 'blur': // add class "blur" to threads or chats with specific chat_id (can be rendered more than one on the page).
            $(`.thread[data-chat-id="${chat_id}"]`).addClass('blur');
            $(`.chat[data-chat-id="${chat_id}"]`).addClass('blur');
            break;
        case 'unblur': // remove class "blur" from threads or chats with specific chat_id (can be rendered more than one on the page).
            $(`.thread[data-chat-id="${chat_id}"]`).removeClass('blur');
            $(`.chat[data-chat-id="${chat_id}"]`).removeClass('blur');
            break;
        case 'report': // add class "reported" (blurred for current user only) to threads or chats with specific chat_id (can be rendered more than one on the page). 
            $(`.thread[data-chat-id="${chat_id}"]`).addClass('reported');
            $(`.chat[data-chat-id="${chat_id}"]`).addClass('reported');
            break;
        case 'unreport':
            $(`.thread[data-chat-id="${chat_id}"]`).removeClass('reported');
            $(`.chat[data-chat-id="${chat_id}"]`).removeClass('reported');
            break;
        default: // For archive, unarchive, delete, and any future unhandled actions so long as removing the chat is the goal.
            const threadDiv = $(`.thread[data-chat-id="${chat_id}"]`);
            const chatDiv = $(`.chat[data-chat-id="${chat_id}"]`);
            if (threadDiv.first().hasClass('thread-container')) {
                threadDiv.empty(); // permanent thread placeholder in extension.
            } else {
                threadDiv.remove();
            }
            chatDiv.remove();
            break;
    }
}
$(document).ready(function() {
    $(document).on('click', '.chat-options-opener', function(e) {
        e.preventDefault();
        if ($('#chat-options-drowdown').length > 0) {
            $('#chat-options-drowdown').remove();
        }
        const $opener = $(this);
        const chatId = $opener.data('chat-id');
        const dataType = $opener.attr('data-type'); // different options for threads vs chats
        const serverURL = getServerURL();
        const threadURL = `${serverURL}/thread/${chatId}`
        const cancelLink = $(`<a href="#" class="list-group-item chat-options-cancel-link"><span class="float-end text-muted small"><i class="bi bi-x-lg"></i> Close</span></a>`);
        const loadingLink = $(`<a href="#" class="list-group-item chat-options-load-link">Loading...</a>`);
        const directLink = $(`<a href="${threadURL}" class="list-group-item chat-options-direct-link" target="_blank"><i class="bi bi-box-arrow-up-right"></i> Open Thread in new Tab</a>`)
        const copyURLLink = $(`<a href="#" data-url="${threadURL}" class="list-group-item chat-options-copy-url-link"><i class="bi bi-share-fill"></i> Share Thread Page</a>`);
        const copyEmbedLink = $(`<a href="#" data-url="${threadURL}?embed=1" class="list-group-item chat-options-copy-embed-link"><i class="bi bi-code-slash"></i> Copy Embed Link</a>`);
        const copyEmbedIFrameLink = $(`<a href="#" data-url="${threadURL}?embed=1" class="list-group-item chat-options-copy-embed-iframe-link"><i class="bi bi-code-slash"></i> Copy Embed Code</a>`);
        cancelLink.on('click', function(e) {
            e.preventDefault();
            $('#chat-options-drowdown').slideUp(300, () => {
                $('#chat-options-drowdown').remove();
            });
        });
        loadingLink.on('click', function(e) {
            e.preventDefault();
            $('.chat-options-load-link').empty().append('Please wait...');
        });
        copyURLLink.on('click', function(e) {
            e.preventDefault();
            const url = $(this).attr('data-url');
            navigator.clipboard.writeText(url).then(function() {
                $('.chat-options-copy-url-link').empty().append('<span class="text text-success">Thread URL Copied!</span>');
            });
        });
        copyEmbedLink.on('click', function(e) {
            e.preventDefault();
            const url = $(this).attr('data-url');
            navigator.clipboard.writeText(url).then(function() {
                $('.chat-options-copy-embed-link').empty().append('<span class="text text-success">Embed Link Copied!</span>');
            });
        });
        copyEmbedIFrameLink.on('click', function(e) {
            e.preventDefault();
            const url = $(this).attr('data-url');
            const iframeCode = `<iframe src="${url}" style="width:100%;height:100%;border:none;outline:none;"></iframe>`;
            navigator.clipboard.writeText(iframeCode).then(function() {
                $('.chat-options-copy-embed-iframe-link').empty().append('<span class="text text-success">Embed Code Copied!</span>');
            });
        });
        const listGroup = $(`<div class="list-group list-group-flush" id="chat-options-list-group"></div>`);
        listGroup.append(cancelLink);
        if (dataType == 'thread') {
            listGroup.append(directLink);
            listGroup.append(copyURLLink);
            listGroup.append(copyEmbedLink);
            listGroup.append(copyEmbedIFrameLink);
        }
        listGroup.append(loadingLink);
        const container = $(`<div class="chat-options-drowdown" id="chat-options-drowdown" tabindex="-1"></div>`);
        container.append(listGroup);
        $opener.parent().after(container);
        if (typeof render_bsi == 'function') {
            render_bsi();
        }
        container.slideDown(300);

        /* 
            Send chatId to /chat_options endpoint to get options for this particular thread or chat.
            Remove .chat-options-load-link and add the options to the #chat-options-list-group.
            Use the key as the data-action attribute and the value as the text of the link.
            Add the chat id to the data-chat-id attribute of each link.
            When each link is clicked, POST chatId and action to /chat_option endpoint.
            Use keys chat_id and perform_action to send the data.
            The button listeners are abstract so that new actions can be added without changing this script.
            The server will return a JSON object with a key of "options" and a value of a dict of options.
            Expected a dict of options like this:
            {
                "block": "Block User", // Available for all users
                "unblock": "Unblock User", // Available for all users
                "blur": "Blur This Chat", // Available if the chat is a reply to the current user
                "unblur": "Unblur This Chat", // Available if the chat is a reply to the current user
                "report": "Report This (Chat/Thread)", // Available for all users, server will determine if it's a chat or thread.
                "archive": "Archive Thread", // Available to the author of a specific thread, but not the author of a chat.
                "delete": "Delete Chat", // Available to the author of a specific chat, not the author of a thread.
            }
            The server will return a JSON object with keys "ok", "chat_id", "author_pk", and "action".
        */
        const jwt = localStorage.getItem('jwt');
        if (!jwt) {
            $('.chat-options-load-link').empty().append('Please login for more options.');
            $('.chat-options-load-link').off('click').on('click', function(e) {
                e.preventDefault();
                $('#chat-options-drowdown').slideUp(300, () => {
                    $('#chat-options-drowdown').remove();
                });
            });
            return;
        }
        $.ajax({
            method: 'GET',
            url: `${serverURL}/api/chat_options/${chatId}`,
            dataType: 'json',
            headers: {
                'Accept': 'application/json',
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${jwt}`
            },
            success: function(data) {
                if (data && data.options && typeof data.options == 'object') {
                    $('.chat-options-load-link').remove();
                    try {
                        const options_list = [];
                        for (const [key, value] of Object.entries(data.options)) {
                            options_list.push([key, value]);
                        }
                        const sorted_options = options_list.sort((a, b) => a[1].localeCompare(b[1]));
                        for (var i = 0; i < sorted_options.length; i++) {
                            if (!sorted_options[i] || !sorted_options[i][0] || !sorted_options[i][1]) {
                                continue;
                            }
                            const key = sorted_options[i][0];
                            const value = sorted_options[i][1];
                            const optionLink = $(
                                `<a href="#" class="list-group-item chat-options-action-link" data-action="${key}" data-chat-id="${chatId}">
                                    ${value}
                                </a>`
                            );
                            // The order of these if statements is important.
                            // Add icons to the links based on the key.
                            if (key.toLowerCase().indexOf('unbookmark') > -1) {
                                optionLink.prepend(bsi('unbookmark')); // bookmark-star-fill
                            } else if (key.toLowerCase().indexOf('bookmark') > -1) {
                                optionLink.prepend(bsi('bookmark')); // bookmark-plus
                            } else if (key.toLowerCase().indexOf('block') > -1) {
                                optionLink.prepend(bsi('block')); // person-slash
                            } else if (key.toLowerCase().indexOf('blur') > -1) {
                                optionLink.prepend(bsi('blur')); // eye-slash
                            } else if (key.toLowerCase().indexOf('report') > -1) {
                                optionLink.prepend(bsi('report')); // flag
                            } else if (key.toLowerCase().indexOf('archive') > -1) {
                                optionLink.prepend(bsi('archive')); // archive-fill
                            } else if (key.toLowerCase().indexOf('delete') > -1) {
                                optionLink.prepend(bsi('delete')); // trash3-fill
                            } else if (key.toLowerCase().indexOf('ban') > -1) {
                                optionLink.prepend(bsi('ban')); // ban
                            }
                            // If the value to lower case contains ", remove", style the link as red.
                            if (value.toLowerCase().indexOf(', remove') > -1) {
                                optionLink.addClass('text-danger');
                            }
                            $('#chat-options-list-group').append(optionLink);
                            optionLink.on('click', function(e) {
                                e.preventDefault();
                                const action = $(this).data('action');
                                const chatId = $(this).data('chat-id');
                                const jwt = localStorage.getItem('jwt');
                                const endpoint = `${serverURL}/api/perform_chat_option`;
                                const payload = {
                                    chat_id: chatId,
                                    perform_action: action
                                };
                                console.log("Chat Option Clicked: ", action, chatId, endpoint, payload);
                                $.ajax({
                                    method: 'POST',
                                    url: endpoint,
                                    dataType: 'json',
                                    headers: {
                                        'Accept': 'application/json',
                                        'Content-Type': 'application/json',
                                        'Authorization': `Bearer ${jwt}`
                                    },
                                    data: JSON.stringify(payload),
                                    success: function(data) {
                                        console.log("Chat Option Result: ", data);
                                        if (data && data.ok) {
                                            try { // action is done on the server, now perform the client-side action
                                                renderActionResult(data.chat_id, data.action, data.author_pk);
                                                // hide+remove the dropdown
                                                $('#chat-options-drowdown').slideUp(300, () => {
                                                    $('#chat-options-drowdown').remove();
                                                });
                                            } catch (e) {
                                                console.error('Error performing action: ', e);
                                                $('.chat-options-cancel-link').addClass('text-danger').empty().append(e.message || 'Error performing action');
                                            }
                                        } else {
                                            console.error('Error performing action, data not ok: ', data);
                                            $('.chat-options-cancel-link').addClass('text-danger').empty().append('Error performing action');
                                        }
                                    },
                                    error: function(xhr, status, error) {
                                        console.error('Error performing action: ', xhr, status, error);
                                        $('.chat-options-cancel-link').addClass('text-danger').empty().append('Error performing action');
                                    }
                                });
                            });
                        }
                        if (typeof render_bsi == 'function') {
                            setTimeout(render_bsi, 50);
                        }
                    } catch (e) {
                        console.error('Error rendering chat options: ', e);
                        $('.chat-options-cancel-link').addClass('text-danger').empty().append(e.message || 'Error rendering options');
                    }
                } else { // No options available
                    $('.chat-options-load-link').empty().append('No actions available.');
                    $('.chat-options-load-link').off('click').on('click', function(e) {
                        e.preventDefault();
                        $('#chat-options-drowdown').slideUp(300, () => {
                            $('#chat-options-drowdown').remove();
                        });
                    });
                }
            },
            error: function(xhr, status, error) {
                // If 401, remind user to login
                if (xhr.status == 401) {
                    $('.chat-options-load-link').empty().append('Please login for more options.');
                    $('.chat-options-load-link').off('click').on('click', function(e) {
                        e.preventDefault();
                        $('#chat-options-drowdown').slideUp(300, () => {
                            $('#chat-options-drowdown').remove();
                        });
                    });
                } else {
                    console.error('Error loading chat options: ', error);
                    $('.chat-options-load-link').empty().append('Error loading options').addClass('text-danger');
                }
            }
        });
    });
});
